
suppressMessages(library(RKernel))
options(width=120)

# ## Preparing the data
# The data we are using are in the file "ESS-relig-CHES.RData".

suppressMessages(library(memisc))
library(magrittr)


load(file="ESS-relig-CHES.RData")


load(file="perc-denom-country-wide.RData")


ESS_work_west <- subset(ESS_pray_denom_CHES,
    country2 %in% c(
        "Austria",
        "Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels",
        "Denmark",
        "France",
        "Germany (West)","Germany/Bavaria",
        "Ireland",
        "Italy",
        "Luxembourg",
        "Netherlands",
        "Norway",
        "Portugal",
        "Spain/Castilia","Spain/Catalunia","Spain/Basque",
        "Sweden",
        "Switzerland",
        "United Kingdom/England",
        "United Kingdom/Scotland",
        "United Kingdom/Wales"
    ) & !is.na(religious_principles),
    select=c(CHES_id,
             year,
             essround,
             denom,
             pray,
             oesch8,
             wgt,
             id,
             country2,
             lrecon,
             galtan,
             spendvtax,
             deregulation,
             redistribution,
             civlib_laworder,
             sociallifestyle,
             religious_principles,
             immigration))


ESS_work_west %<>% as.data.frame()

# The information about religious denomination appears to be incorrect or misleading 
# in France until 2005 and in the United Kingdom from 2004 to 2007 and after 2020.

options(contrasts=c("contr.sum","contr.sum"))


ESS_work_west %<>% within({
    Country <- recode(as.character(country2),
                      c("Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels") -> "Belgium",
                      c("Germany (West)","Germany/Bavaria") -> "Germany",
                      c("Spain/Castilia","Spain/Catalunia","Spain/Basque") -> "Spain",
                      c("United Kingdom/England","United Kingdom/Scotland","United Kingdom/Wales") 
                          -> "United Kingdom",
                      copy=TRUE)
    Country <- factor(Country,levels=sort(levels(Country)))
    denom[Country=="France" & year <= 2005 | 
         Country=="United Kingdom" & year >= 2004 & year <= 2007] <- NA
    cathprot <- recode(as.numeric(denom),
                      "Catholic"     <- 1,
                      "Protestant"   <- 2
                     )
})


ESS_work_west <- merge(ESS_work_west, perc.denom.country.wide,
                       by.x = 'Country',
                       by.y = 'country',
                       all.x = TRUE,
                       all.y = FALSE)


ESS_work_west %<>% within({
    foreach(x = c(lrecon,
                  galtan,
                  spendvtax,
                  deregulation,
                  redistribution,
                  civlib_laworder,
                  sociallifestyle,
                  religious_principles,
                  immigration),
            y = c(lrecon.sq,
                  galtan.sq,
                  spendvtax.sq,
                  deregulation.sq,
                  redistribution.sq,
                  civlib_laworder.sq,
                  sociallifestyle.sq,
                  religious_principles.sq,
                  immigration.sq),
            {
                x <- (x-5)/10
                y <- x^2
            })
    oesch8 <- factor(oesch8,levels=levels(oesch8)[c(3,1:2,4:8)]) # Production workers as baseline
    country <- factor(country2) # Drop countries not included in this subset
    time <- (year - 2010)/10
    contrasts(cathprot) <- contr("sum", base = 1)
    contrasts(oesch8) <- contr("sum", base = 1)
    contrasts(pray) <- contr("sum", base = 1)
    No_religion <- (No_religion - 50)/100
    Denom_Het <- 2*(1-(Cath_Prot/100)^2-(1-Cath_Prot/100)^2)
    Cath_Prot <- (Cath_Prot - 50)/100
    })

# ## Model estimation

library(mclogit)


model_relprin_full <- mclogit(wgt|id~religious_principles.sq+
                                immigration.sq+
                                lrecon.sq+
                                religious_principles/(cathprot)*(pray)*time+
                                religious_principles/(cathprot)*(No_religion+Cath_Prot+Denom_Het)+
                                religious_principles/oesch8*time+
                                immigration/(oesch8+pray+cathprot)*time+
                                lrecon/(oesch8+pray+cathprot)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )


# ## Wald tests

source("Wald.R")


(Wald_relprin_full <- 
 Wald(model_relprin_full,
     list(~religious_principles.sq,
          ~religious_principles,
          ~religious_principles:time,
          ~religious_principles:cathprot,
          ~religious_principles:cathprot:time,
          ~religious_principles:pray,
          ~religious_principles:pray:time,
          ~religious_principles:cathprot:pray,
          ~religious_principles:cathprot:pray:time,
          ~religious_principles:No_religion,
          ~religious_principles:cathprot:No_religion,
          ~religious_principles:Cath_Prot,
          ~religious_principles:cathprot:Cath_Prot,
          ~religious_principles:Denom_Het,
          ~religious_principles:cathprot:Denom_Het
           )))


Wald_relprin_full %>%
        dimrename(          
        "religious_principles.sq" = "Religious Principles squared",
        "religious_principles" = "Religious Principles",
        "religious_principles:time" = "$\\times$ time",
        "religious_principles:cathprot" = "$\\times$ Catholic/Protestant",
        "religious_principles:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        "religious_principles:pray" = "$\\times$ freq of prayer",
        "religious_principles:pray:time" = "$\\times$ freq of prayer $\\times$ time",
        "religious_principles:cathprot:pray" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer",
        "religious_principles:cathprot:pray:time" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer $\\times$ time",
        "religious_principles:No_religion" = "$\\times$ secular",
        "religious_principles:cathprot:No_religion" = "$\\times$ Catholic/Protestant $\\times$ secular",
        "religious_principles:Cath_Prot" = "$\\times$ denom. balance",
        "religious_principles:cathprot:Cath_Prot" = "$\\times$ Catholic/Protestant $\\times$ denom. balance",
        "religious_principles:Denom_Het" = "$\\times$ denom. heterogeneity",
        "religious_principles:cathprot:Denom_Het" = "$\\times$ Catholic/Protestant $\\times$ denom. heterogeneity"
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="cathprot-context-relprin/Wald-cathprot-context-relprin-full.tex")

# ## Reduced model

model_relprin <- mclogit(wgt|id~religious_principles.sq+
                                immigration.sq+
                                lrecon.sq+
                                religious_principles/(cathprot)*(pray)*time+
                                religious_principles/(cathprot)*(Cath_Prot+Denom_Het)+
                                religious_principles/oesch8*time+
                                immigration/(oesch8+pray+cathprot)*time+
                                lrecon/(oesch8+pray+cathprot)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )



(Wald_relprin <- 
 Wald(model_relprin,
     list(~religious_principles.sq,
          ~religious_principles,
          ~religious_principles:time,
          ~religious_principles:cathprot,
          ~religious_principles:cathprot:time,
          ~religious_principles:pray,
          ~religious_principles:pray:time,
          ~religious_principles:cathprot:pray,
          ~religious_principles:cathprot:pray:time,
          ~religious_principles:Cath_Prot,
          ~religious_principles:cathprot:Cath_Prot,
          ~religious_principles:Denom_Het,
          ~religious_principles:cathprot:Denom_Het
           )))


Wald_relprin %>%
        dimrename(          
        "religious_principles.sq" = "Religious Principles squared",
        "religious_principles" = "Religious Principles",
        "religious_principles:time" = "$\\times$ time",
        "religious_principles:cathprot" = "$\\times$ Catholic/Protestant",
        "religious_principles:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        "religious_principles:pray" = "$\\times$ freq of prayer",
        "religious_principles:pray:time" = "$\\times$ freq of prayer $\\times$ time",
        "religious_principles:cathprot:pray" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer",
        "religious_principles:cathprot:pray:time" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer $\\times$ time",
        "religious_principles:Cath_Prot" = "$\\times$ denom. balance",
        "religious_principles:cathprot:Cath_Prot" = "$\\times$ Catholic/Protestant $\\times$ denom. balance",
        "religious_principles:Denom_Het" = "$\\times$ denom. heterogeneity",
        "religious_principles:cathprot:Denom_Het" = "$\\times$ Catholic/Protestant $\\times$ denom. heterogeneity"
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="cathprot-context-relprin/Wald-cathprot-context-relprin.tex")


(Wald_relprin_short <- 
 Wald(model_relprin,
     list(~religious_principles.sq,
          ~religious_principles,
          ~religious_principles:time,
          ~religious_principles:cathprot,
          ~religious_principles:cathprot:time
           )))


Wald_relprin_short %>%
        dimrename(          
        "religious_principles.sq" = "Religious Principles squared",
        "religious_principles" = "Religious Principles",
        "religious_principles:time" = "$\\times$ time",
        "religious_principles:cathprot" = "$\\times$ Catholic/Protestant",
        "religious_principles:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="cathprot-context-relprin/Wald-cathprot-context-relprin-short.tex")


save(model_relprin,model_relprin_full,file="model-cathprot-prayer-relprin-context.RData")

